<?php

namespace WeDevs\DokanPro\Modules\PayPalMarketplace\Blocks;

use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;
use WeDevs\DokanPro\Modules\PayPalMarketplace\Helper;

class CartCheckoutBlockSupport extends AbstractPaymentMethodType {

    /**
	 * The block supported gateway instance.
	 *
	 * @var \WeDevs\DokanPro\Modules\PayPalMarketplace\PaymentMethods\PayPal
	 */
	private $gateway;

    /**
     * Payment method name defined by payment methods extending this class.
     *
     * @since DOKAN_SINCE
     *
     * @var string
     */
    protected $name;

    /**
     * Payment method handle.
     *
     * @since DOKAN_SINCE
     *
     * @var string
     */
    protected $handle;

    /**
     * Class constructor.
     *
     * @since 3.15.0
     *
     * @return void
     */
	public function __construct() {
        $this->name = Helper::get_gateway_id();
        $this->handle = $this->name . '_blocks';

        $gateways       = WC()->payment_gateways->payment_gateways();
		$this->gateway  = $gateways[ $this->name ];
	}

    /**
     * This function will get called during the server side initialization process and is a good place to put any settings population etc.
     * Basically anything you need to do to initialize your gateway. Note, this will be called on every request so don't put anything expensive here.
     *
     * @since 3.15.0
     *
     * @return void
     */
    public function initialize() {
        $this->settings = Helper::get_settings();
    }

    /**
     * This should return whether the payment method is active or not.
     *
     * @since 3.15.0
     *
     * @return bool
     */
    public function is_active(): bool {
        return Helper::is_ready();
    }

    /**
     * Returns an array of scripts/handles to be registered for this payment method.
     *
     * @see https://github.com/woocommerce/woocommerce-blocks/blob/060f63c04f0f34f645200b5d4da9212125c49177/docs/third-party-developers/extensibility/checkout-payment-methods/payment-method-integration.md#registering-assets
     *
     * @since 3.15.0
     *
     * @return array
     */
    public function get_payment_method_script_handles(): array {
        $asset_path   = DOKAN_PAYPAL_MP_PATH . 'assets/blocks/payment-support/index.asset.php';
        $version      = dokan_pro()->version;

        $dependencies = [];

        if ( file_exists( $asset_path ) ) {
            $asset        = require $asset_path;
            $version      = $asset['version'] ?? $version;
            $dependencies = $asset['dependencies'] ?? $dependencies;
        }

        wp_register_script(
            $this->handle,
            DOKAN_PAYPAL_MP_ASSETS . 'blocks/payment-support/index.js',
            $dependencies,
            $version,
            true
        );

        return [ $this->handle ];
    }

    /**
     * Return associative array of data you want to be exposed for your payment method client side.
     * This data will be available client side via wc.wcSettings.getSetting.
     *
     * @since 3.15.0
     *
     * @return array
     */
    public function get_payment_method_data(): array {
	    $payment_fields_data['title']               = $this->get_setting( 'title', __( 'Dokan PayPal Marketplace', 'dokan' ) );
	    $payment_fields_data['sandbox_mode']        = 'yes' === $this->get_setting( 'test_mode' );
	    $payment_fields_data['enabled']             = 'yes' === $this->get_setting( 'enabled' );
	    $payment_fields_data['clientId']            = Helper::get_client_id();
	    $payment_fields_data['description']         = $this->get_setting( 'description', '' );
	    $payment_fields_data['userId']              = is_user_logged_in() ? get_current_user_id() : 0;
        $payment_fields_data['supports']            = array_filter( $this->gateway->supports, [ $this->gateway, 'supports' ] );

		return $payment_fields_data;
    }
}
